<?php

declare(strict_types=1);

namespace Erlage\Photogram\Data\Dtos\User;

use Erlage\Photogram\Settings;
use Erlage\Photogram\Data\Dtos\AbstractDTO;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Constants\SystemConstants;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Data\Dtos\Traits\TraitComplaintTransformerDTO;

final class UserMetaPushSettingsDTO extends AbstractDTO
{
    /**
     * to comply with DTO interface
     */
    use TraitComplaintTransformerDTO;

    /*
    |--------------------------------------------------------------------------
    | available settings key map
    |--------------------------------------------------------------------------
    */

    // master setting, to disable all

    const KEY_PAUSE_ALL = 'pause_all';

    // likes on posts of user

    const KEY_LIKES = 'likes';

    // comments on posts of user

    const KEY_COMMENTS = 'comments';

    // someone tags you in their photo

    const KEY_PHOTOS_OF_YOU = 'photos_of_you';

    // likes on posts in which user is tagged

    const KEY_LIKES_ON_PHOTOS_OF_YOU = 'likes_on_photos_of_you';

    // comments on posts in which user is tagged

    const KEY_COMMENTS_ON_PHOTOS_OF_YOU = 'comments_on_photos_of_you';

    // likes on comments by user

    const KEY_COMMENT_LIKES_AND_PINS = 'comment_likes_and_pins';

    // notify when someone accepted your follow request

    const KEY_ACCEPTED_FOLLOW_REQUEST = 'accepted_follow_request';

    /*
    |--------------------------------------------------------------------------
    | value types
    |--------------------------------------------------------------------------
    */

    const VALUE_NOT_AVAILABLE = 'not_available';

    const VALUE_ON = 'on';

    const VALUE_OFF = 'off';

    const VALUE_FROM_EVERYONE = 'from_everyone';

    const VALUE_FROM_PEOPLE_I_FOLLOW = 'from_people_i_follow';

    /*
    |--------------------------------------------------------------------------
    | data
    |--------------------------------------------------------------------------
    */

    /**
     * @var string
     */
    private $pauseAll;

    /**
     * @var string
     */
    private $likes;

    /**
     * @var string
     */
    private $comments;

    /**
     * @var string
     */
    private $photosOfYou;

    /**
     * @var string
     */
    private $likesOnPhotosOfYou;

    /**
     * @var string
     */
    private $commentsOnPhotosOfYou;

    /**
     * @var string
     */
    private $commentLikesAndPins;

    /**
     * @var string
     */
    private $acceptedFollowRequest;

    /*
    |--------------------------------------------------------------------------
    | getters
    |--------------------------------------------------------------------------
    */

    public function getPauseAll(): string
    {
        return $this -> pauseAll;
    }

    public function getLikes(): string
    {
        return $this -> likes;
    }

    public function getComments(): string
    {
        return $this -> comments;
    }

    public function getPhotosOfYou(): string
    {
        return $this -> photosOfYou;
    }

    public function getLikesOnPhotosOfYou(): string
    {
        return $this -> likesOnPhotosOfYou;
    }

    public function getCommentsOnPhotosOfYou(): string
    {
        return $this -> commentsOnPhotosOfYou;
    }

    public function getCommentLikesAndPins(): string
    {
        return $this -> commentLikesAndPins;
    }

    public function getAcceptedFollowRequest(): string
    {
        return $this -> acceptedFollowRequest;
    }

    /*
    |--------------------------------------------------------------------------
    | setters
    |--------------------------------------------------------------------------
    */

    /**
     * @return static 
     */
    public function setPauseAll(string $pauseAll)
    {
        $this -> pauseAll = $pauseAll;

        return $this;
    }

    /**
     * @return static
     */
    public function setLikes(string $likes)
    {
        $this -> likes = $likes;

        return $this;
    }

    /**
     * @return static 
     */
    public function setComments(string $comments)
    {
        $this -> comments = $comments;

        return $this;
    }

    /**
     * @return static 
     */
    public function setPhotosOfYou(string $photosOfYou)
    {
        $this -> photosOfYou = $photosOfYou;

        return $this;
    }

    /**
     * @return static 
     */
    public function setLikesOnPhotosOfYou(string $likesOnPhotosOfYou)
    {
        $this -> likesOnPhotosOfYou = $likesOnPhotosOfYou;

        return $this;
    }

    /**
     * @return static 
     */
    public function setCommentsOnPhotosOfYou(string $commentsOnPhotosOfYou)
    {
        $this -> commentsOnPhotosOfYou = $commentsOnPhotosOfYou;

        return $this;
    }

    /**
     * @return static 
     */
    public function setCommentLikesAndPins(string $commentLikesAndPins)
    {
        $this -> commentLikesAndPins = $commentLikesAndPins;

        return $this;
    }

    /**
     * @return static 
     */
    public function setAcceptedFollowRequest(string $acceptedFollowRequest)
    {
        $this -> acceptedFollowRequest = $acceptedFollowRequest;

        return $this;
    }

    /*
    |--------------------------------------------------------------------------
    | create from default settings
    |--------------------------------------------------------------------------
    */

    public static function fromDefaultSettings(): self
    {
        return (new self())
            -> setPauseAll(
                Settings::getString(
                    ServerConstants::SS_ENUM_NOTIFICATION_PAUSE_ALL
                )
            )
            -> setLikes(
                Settings::getString(
                    ServerConstants::SS_ENUM_NOTIFICATION_LIKES
                )
            )
            -> setComments(
                Settings::getString(
                    ServerConstants::SS_ENUM_NOTIFICATION_COMMENTS
                )
            )
            -> setPhotosOfYou(
                Settings::getString(
                    ServerConstants::SS_ENUM_NOTIFICATION_PHOTOS_OF_YOU
                )
            )
            -> setLikesOnPhotosOfYou(
                Settings::getString(
                    ServerConstants::SS_ENUM_NOTIFICATION_LIKES_ON_PHOTOS_OF_YOU
                )
            )
            -> setCommentsOnPhotosOfYou(
                Settings::getString(
                    ServerConstants::SS_ENUM_NOTIFICATION_COMMENTS_ON_PHOTOS_OF_YOU
                )
            )
            -> setCommentLikesAndPins(
                Settings::getString(
                    ServerConstants::SS_ENUM_NOTIFICATION_COMMENT_LIKES_AND_PINS
                )
            )
            -> setAcceptedFollowRequest(
                Settings::getString(
                    ServerConstants::SS_ENUM_NOTIFICATION_ACCEPTED_FOLLOW_REQUEST
                )
            );
    }

    /*
    |--------------------------------------------------------------------------
    | from json
    |--------------------------------------------------------------------------
    */

    public static function fromJsonMap(array $data): self
    {
        return (new static())
            -> setPauseAll((string) $data[self::KEY_PAUSE_ALL])
            -> setLikes((string) $data[self::KEY_LIKES])
            -> setComments((string) $data[self::KEY_COMMENTS])
            -> setPhotosOfYou((string) $data[self::KEY_PHOTOS_OF_YOU])
            -> setLikesOnPhotosOfYou((string) $data[self::KEY_LIKES_ON_PHOTOS_OF_YOU])
            -> setCommentsOnPhotosOfYou((string) $data[self::KEY_COMMENTS_ON_PHOTOS_OF_YOU])
            -> setCommentLikesAndPins((string) $data[self::KEY_COMMENT_LIKES_AND_PINS])
            -> setAcceptedFollowRequest((string) $data[self::KEY_ACCEPTED_FOLLOW_REQUEST]);
    }

    /*
    |--------------------------------------------------------------------------
    | serialization for json_encode
    |--------------------------------------------------------------------------
    */

    public function jsonSerialize()
    {
        return array(
            self::KEY_PAUSE_ALL                    => $this -> pauseAll,
            self::KEY_LIKES                        => $this -> likes,
            self::KEY_COMMENTS                     => $this -> comments,
            self::KEY_PHOTOS_OF_YOU                => $this -> photosOfYou,
            self::KEY_LIKES_ON_PHOTOS_OF_YOU       => $this -> likesOnPhotosOfYou,
            self::KEY_COMMENTS_ON_PHOTOS_OF_YOU    => $this -> commentsOnPhotosOfYou,
            self::KEY_COMMENT_LIKES_AND_PINS       => $this -> commentLikesAndPins,
            self::KEY_ACCEPTED_FOLLOW_REQUEST      => $this -> acceptedFollowRequest,
        );
    }

    /*
    |--------------------------------------------------------------------------
    | implement DTO methods
    |--------------------------------------------------------------------------
    */

    /**
     * @param UserMetaPushSettingsDTO $userAccountSettingsDTO 
     */
    public static function validate($userAccountSettingsDTO): string
    {
        if (
            ! \in_array(
                $userAccountSettingsDTO -> getPauseAll(),
                array(
                    self::VALUE_OFF, self::VALUE_ON,
                )
            )
        ) {
            return ResponseConstants::ERROR_BAD_REQUEST_MSG;
        }

        if (
            ! \in_array(
                $userAccountSettingsDTO -> getLikes(),
                array(
                    self::VALUE_OFF, self::VALUE_FROM_PEOPLE_I_FOLLOW, self::VALUE_FROM_EVERYONE,
                )
            )
        ) {
            return ResponseConstants::ERROR_BAD_REQUEST_MSG;
        }

        if (
            ! \in_array(
                $userAccountSettingsDTO -> getComments(),
                array(
                    self::VALUE_OFF, self::VALUE_FROM_PEOPLE_I_FOLLOW, self::VALUE_FROM_EVERYONE,
                )
            )
        ) {
            return ResponseConstants::ERROR_BAD_REQUEST_MSG;
        }

        if (
            ! \in_array(
                $userAccountSettingsDTO -> getPhotosOfYou(),
                array(
                    self::VALUE_OFF, self::VALUE_FROM_PEOPLE_I_FOLLOW, self::VALUE_FROM_EVERYONE,
                )
            )
        ) {
            return ResponseConstants::ERROR_BAD_REQUEST_MSG;
        }

        if (
            ! \in_array(
                $userAccountSettingsDTO -> getLikesOnPhotosOfYou(),
                array(
                    self::VALUE_OFF, self::VALUE_ON,
                )
            )
        ) {
            return ResponseConstants::ERROR_BAD_REQUEST_MSG;
        }

        if (
            ! \in_array(
                $userAccountSettingsDTO -> getCommentsOnPhotosOfYou(),
                array(
                    self::VALUE_OFF, self::VALUE_ON,
                )
            )
        ) {
            return ResponseConstants::ERROR_BAD_REQUEST_MSG;
        }

        if (
            ! \in_array(
                $userAccountSettingsDTO -> getCommentLikesAndPins(),
                array(
                    self::VALUE_OFF, self::VALUE_ON,
                )
            )
        ) {
            return ResponseConstants::ERROR_BAD_REQUEST_MSG;
        }

        if (
            ! \in_array(
                $userAccountSettingsDTO -> getAcceptedFollowRequest(),
                array(
                    self::VALUE_OFF, self::VALUE_ON,
                )
            )
        ) {
            return ResponseConstants::ERROR_BAD_REQUEST_MSG;
        }

        return SystemConstants::OK;
    }

    /**
     * @param UserMetaPushSettingsDTO $notificationTypeDTO 
     */
    public static function serialize($notificationTypeDTO): string
    {
        return \json_encode($notificationTypeDTO);
    }

    /**
     * @return UserMetaPushSettingsDTO 
     */
    public static function deSerialize(string $displayItemDTO)
    {
        $data = \json_decode($displayItemDTO, true);

        return self::fromJsonMap($data);
    }

    /*
    |--------------------------------------------------------------------------
    | helper getters
    |--------------------------------------------------------------------------
    */

    public function arePaused(): bool
    {
        return $this -> pauseAll == self::VALUE_ON;
    }
}
